/*
 *   SPECjEnterprise2010 - a benchmark for enterprise middleware
 *  Copyright 1995-2010 Standard Performance Evaluation Corporation
 *   All Rights Reserved
 *
 * This source code is provided as is, without any express or implied warranty.
 *
 *  History:
 *  Date        ID, Company               Description
 *  ----------  ----------------          ----------------------------------------------
 *  2003/01/01  John Stecher, IBM         Created for SPECjAppServer2004
 *  2003/04/15  John Stecher, IBM         added debugging
 *  2003/04/20  John Stecher, IBM         optimized lookup of itemDataBeans
 *  2003/04/20  John Stecher, IBM         Explicitly remove session beans after use
 *  2003/04/24  John Stecher, IBM         Updated doPurchase to return a ArrayList because we
 *                                        are now using the session bean to return the ArrayList
 *                                        as a wrapper around the items, and the orderDataBean
 *  2003/04/29  John Stecher, IBM         Updated to use javax.rmi.PortableRemoteObject.narrow
 *  2003/05/01  John Stecher, IBM         Imported Russ Raymundo's mfg side of the benchmark for web base driving
 *  2003/05/05  John Stecher, IBM         Made changes to catch noSuchObject exception on remove and display
 *                                        warning message.
 *  2003/05/05  John Stecher, IBM         Made changes to ensure that users are updated when car is sold out from
 *                                        under them with a warning message.
 *  2003/05/06  John Stecher, IBM         Made changes to allow drive instead of application to determine
 *                                        if an order is to be deferred or added to inventory immediately
 *  2003/06/25  John Stecher, IBM         Made changes to this code in line with best practices
 *  2003/08/30  John Stecher, IBM         Updated for new sell functionality
 *  2003/10/27  John Stecher, IBM         Updated to store ItemBrowserSes handle in HTTP Session instead of bean
 *  2003/11/26  Tom Daly, Sun             Added support for category to getVehicleQuotes()
 *  2003/12/05  John Stecher, IBM         Added Atomicity Tests
 *  2004/01/08  John Stecher, IBM         Changed code to eliminate unused objects being passed into methods
 *  2004/01/15  John Stecher, IBM         Changes to make Atomicity Tests work in remote case (see osgjava-6324)
 *  2004/01/25  John Stecher, IBM         Added code for cache consistancy test.
 *  2004/02/18  Samuel Kounev, Darmstadt  Modified updateItemPrice to call OrderAuditSes instead of OrderSes.
 *  2004/02/27  Samuel Kounev, Darmstadt  Fixed a bug in atomicityTestTwo: call to doPurchase should catch
 *                                        SPECjAppServerException and not Exception.
 *  2009/05/08  Anoop Gupta, Oracle       Added location parameter to operations on WorkOrder as 
 *                                        WorkOrder should be looked up at specific location
 *  2009/08/09 Tom Daly, Sun              Added getItemJDBC and refactored getItem to getItemJPA
 *  2009/08/09 Tom Daly, Sun              removed getItemJDBC and added updateItemJDBC
 */
package org.spec.jent.servlet.helper;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Calendar;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import javax.naming.NamingException;
import javax.servlet.http.HttpSession;

import org.spec.jent.ejb.mfg.entity.WorkOrder;
import org.spec.jent.ejb.mfg.session.MfgService;
import org.spec.jent.ejb.mfg.session.WorkOrderService;
import org.spec.jent.ejb.orders.entity.Customer;
import org.spec.jent.ejb.orders.entity.CustomerInventory;
import org.spec.jent.ejb.orders.entity.Item;
import org.spec.jent.ejb.orders.entity.Order;
import org.spec.jent.ejb.orders.session.CustomerService;
import org.spec.jent.ejb.orders.session.InsufficientCreditException;
import org.spec.jent.ejb.orders.session.ItemBrowserService;
import org.spec.jent.ejb.orders.session.OrderAuditService;
import org.spec.jent.ejb.orders.session.OrderService;
import org.spec.jent.ejb.shoppingcart.entity.ShoppingCart;
import org.spec.jent.ejb.supplier.entity.Supplier;
import org.spec.jent.ejb.supplier.session.SupplierService;
import org.spec.jent.loader.DatabaseHelper;

/**
 * Provides the generic client side access to each of the SpecJ operations such
 * as login, logout, view inventory, shopping cart etc.
 * The SpecAction class handles generic client side processing for each
 * operation type such as input verification, etc.
 * The SpecAction class does not handle user interface processing and should be
 * used by a class that is UI specific. For example, SpecServletAction manages
 * and builds a a web interface to SpecJ, making calls to SpecAction methods to
 * actually perform each operation.
 *
 */
public class SpecAction implements SpecServices {
    /*
     * This atomicity test will attempt to drive a sell operation on a
     * dealer inventory,
     * the cars will be removed from the inventory, and then upon attempting to
     * update the dealer's financial information an exception will be thrown.
     * The count of inventory items before and after
     * the transaction should remain the same for the transaction to be atomic.
     */

    public boolean atomicityTestThree() throws Exception {
        CustomerService cs = SpecBeanFactory.getService(CustomerService.class);
        OrderService os = SpecBeanFactory.getService(OrderService.class);
        ItemBrowserService ibs = SpecBeanFactory.getService(ItemBrowserService.class);

        BigDecimal creditLimit = new BigDecimal("200000.00");
        Customer customer = cs.selectCustomerWithGoodCredit(creditLimit);
        int custID = customer.getId();

        ShoppingCart sc = new ShoppingCart();

        Collection<Item> itemsColl = ibs.getItems(0, null);
        Iterator<Item> itr = itemsColl.iterator();
        sc.addItem(itr.next(), 25);
        sc.addItem(itr.next(), 1);

        // Retrieve out all information needed to compare before and after states
        Long orderCountInit = os.getOrderCount(custID);
        Collection<CustomerInventory> hColl = this.getHoldings(custID);
        int numOfHoldingsInit = hColl.size();
        BigDecimal balanceInit = customer.getBalance();

        doPurchase(custID, sc, 0, false, true);

        // Retrieve out all information needed to compare before and after states
        Long orderCountAfter = os.getOrderCount(custID);
        hColl = this.getHoldings(custID);
        int numOfHoldingsAfter = hColl.size();
        BigDecimal balanceAfter = customer.getBalance();

        // Remove all session beans used
        return orderCountInit.intValue() == orderCountAfter.intValue() && balanceAfter.compareTo(balanceInit) == 0 && numOfHoldingsInit == numOfHoldingsAfter;
    }

    public boolean atomicityTestTwo() throws Exception {
        CustomerService cs = SpecBeanFactory.getService(CustomerService.class);
        OrderService os = SpecBeanFactory.getService(OrderService.class);
        ItemBrowserService ibs = SpecBeanFactory.getService(ItemBrowserService.class);

        BigDecimal creditLimit = new BigDecimal("200000.00");
        Customer customer = cs.selectCustomerWithGoodCredit(creditLimit);
        int customerId = customer.getId();

        ShoppingCart sc = new ShoppingCart();
        int category = 0;
        Iterator<Item> items = ibs.getItems(category, null).iterator();
        Item itemToAddToCart = (Item) items.next();
        sc.addItem(itemToAddToCart, 1);

        // Retrieve out all information needed to compare before and after states
        Long orderCountInit = os.getOrderCount(customerId);
        Collection<CustomerInventory> hColl = customer.getInventories();
        int numOfHoldingsInit = hColl.size();
        BigDecimal balanceInit = cs.getCustomer(customerId).getBalance();

        doPurchase(customerId, sc, 0, false, false);

        // Retrieve out all information needed to compare before and after states
        Long orderCountAfter = os.getOrderCount(customerId);
        hColl = getHoldings(customerId);
        int numOfHoldingsAfter = hColl.size();
        BigDecimal balanceAfter = cs.getCustomer(customerId).getBalance();

        return orderCountInit.intValue() < orderCountAfter.intValue() && balanceAfter.compareTo(balanceInit) == -1 && numOfHoldingsInit < numOfHoldingsAfter;
    }

    public boolean atomicityTestOne() throws Exception {

        CustomerService cs = SpecBeanFactory.getService(CustomerService.class);
        OrderService os = SpecBeanFactory.getService(OrderService.class);
        ItemBrowserService ibs = SpecBeanFactory.getService(ItemBrowserService.class);

        BigDecimal creditLimit = new BigDecimal("200000.00");
        Customer customer = cs.selectCustomerWithGoodCredit(creditLimit);
        int custID = customer.getId();

        ShoppingCart sc = new ShoppingCart();
        int category = 0;
        Iterator<Item> items = ibs.getItems(category, null).iterator();
        Item itemToAddToCart = (Item) items.next();
        sc.addItem(itemToAddToCart, 1);

        // Retrieve out all information needed to compare before and after states
        Long orderCountInit = os.getOrderCount(custID);
        Collection<CustomerInventory> hColl = this.getHoldings(custID);
        int numOfHoldingsInit = hColl.size();
        BigDecimal balanceInit = cs.getCustomer(custID).getBalance();

        // Drive the actual Order
        doPurchase(custID, sc, 0, false, true);

        // Retrieve out all information needed to compare before and after states
        Long orderCountAfter = os.getOrderCount(custID);
        hColl = this.getHoldings(custID);
        int numOfHoldingsAfter = hColl.size();
        BigDecimal balanceAfter = cs.getCustomer(custID).getBalance();

        return orderCountInit.intValue() == orderCountAfter.intValue() && balanceAfter.compareTo(balanceInit) == 0 && numOfHoldingsInit == numOfHoldingsAfter;
    }

    /**
     * Sell a vehicle and removed the vehicle and quantity for the given dealership.
     * Given a vehicle, remove the inventory, credit the dealerships account,
     * and remove the vehicles from the dealership's portfolio.
     *
     * @param userID the customer requesting the sell
     * @param holdingID the vehicle to be sold
     * @return boolean indicating if the sale occured without error
     */
    public boolean sell(int userID, long holdingID, boolean rollback) {
        CustomerService cs = SpecBeanFactory.getService(CustomerService.class);
        boolean sold = cs.sellInventory(userID, holdingID, rollback);
        return sold;
    }

    /**
     * Cancel an open order and remove the order from the dealers list of
     * orders to be processed in the future.
     * Given a orderID cancel the customers order.
     *
     * @param userID the customer requesting the sell
     * @return boolean true if the order is canceled, false otherwise.
     */
    public boolean cancel(int orderID) {
        OrderService os = SpecBeanFactory.getService(OrderService.class);
        return os.cancelOrder(orderID);
    }

    /**
     * Get the collection of all open orders for a given account
     *
     * @param userID the customer account to retrieve orders for
     * @return Collection OrderDataBeans providing detailed order information
     */
    public List<Order> getOpenOrders(int userID) {
        OrderService os = SpecBeanFactory.getService(OrderService.class);
        return os.getOpenOrders(userID);
    }

    /**
     * Gets a list of the current vehicles either specified in the search or
     * instead just the generic search of all vehicle.  Displays them 10 at a time
     *
     * @param vehciles - an arraylist containing all vehicles to search for in the manufacturers inventory
     * @param session - the http session that is used to hold a reference to the stateful session bean containing the inventory
     * @param browse - an indicator to the application on which direction the dealer want to browse the inventory
     * @param category - indicates category of vehicles to browse in or purchase from
     * @return the VehicleSearch - a wrapper object containing all the vehcile to display on the current page
     */
    public VehicleSearch getVehicleQuotes(List<String> vehicles,
            HttpSession session, String browse, int category) {
        // check if we have a reference to the SF bean
        ItemBrowserService ibs = (ItemBrowserService) session.getAttribute("ibs");
        if (ibs == null) {
            ibs = (ItemBrowserService) SpecBeanFactory.getNewService(ItemBrowserService.class);
            session.setAttribute("ibs", ibs);
        }
        VehicleSearch vs = new VehicleSearch();
        if (browse == null) {
            vs.vehicles = ibs.getItems(category, vehicles);
        } else if (browse.equalsIgnoreCase("bkwd")) {
            vs.vehicles = ibs.browseReverse();
        } else if (browse.equalsIgnoreCase("fwd")) {
            vs.vehicles = ibs.browseForward();
        } else {
            vs.vehicles = ibs.getItems(category, vehicles);
        }
        vs.min = ibs.getCurrentMin();
        vs.max = ibs.getCurrentMax();
        vs.total = ibs.getTotalItems();
        return vs;
    }

    /**
     * Return the dealer's current inventory in stock given a customer ID.
     *
     * @param userID the customer requesting the portfolio
     * @return Collection of the users portfolio of stock holdings
     */
    public Collection<CustomerInventory> getHoldings(int userID) {
        CustomerService cs = SpecBeanFactory.getService(CustomerService.class);
        Collection<CustomerInventory> inventory = cs.getInventories(userID);
        return inventory;
    }

    /**
     * used by the first part of the cache validity test to retrieve an item which will have it's price updated.
     * @param itemID
     * @return
     */
    public Item getItemJPA(String itemID) {
        OrderService orderService = SpecBeanFactory.getService(OrderService.class);
        return orderService.getItem(itemID);
    }

    /**
     * Used by the benchmark auditing to update the price of the item
     * @param itemID
     * @param newval
     * @throws SQLException
     * @throws NamingException
     */
    public void updateItemJDBC(String itemID, BigDecimal newval) throws SQLException, NamingException {
        final Connection c = DatabaseHelper.getConnection(DatabaseHelper.ORDER_KEY);

        c.setAutoCommit(true);
        try {
            final PreparedStatement ps =
                 c.prepareStatement("update O_ITEM SET I_PRICE = " + newval + ", I_VERSION = I_VERSION + 1 WHERE I_ID = '" + itemID + "'");
            try {
                ps.execute();
            } finally {
                ps.close();
            }
        } finally {
            c.close();
        }
    }

    /**
     * Return an CustomerDataBean object for userID describing the dealerships account
     *
     * @param userID the dealership userID to lookup
     * @return dealerships CustomerDataBean
     */
    public Customer getCustomerData(int userID) {
        CustomerService cs = SpecBeanFactory.getService(CustomerService.class);
        Customer customer = cs.getCustomer(userID);
        if (customer == null) {
            throw new IllegalArgumentException("Customer id=" + userID + " not found");
        }
        return customer;
    }

    /**
     * Purchase the items currently in the shopping cart.
     *
     * @param cart the shopping cart that contains the items the customer
     * wishes to purchase
     * @return Order showing the information about the placed order
     */
    public Order doPurchase(int custID, ShoppingCart cart, int location, boolean deferred,
            boolean rollback) throws InsufficientCreditException {
        OrderService os = SpecBeanFactory.getService(OrderService.class);
        return os.newOrder(custID, cart, location, deferred, rollback);
    }

    /**
     * Attempt to authenticate and login a user
     *
     * @param userID the customer to login
     * @return User account data in AccountDataBean
     */
    public boolean login(int userID) {
        CustomerService cs = SpecBeanFactory.getService(CustomerService.class);
        return cs.validateCustomer(userID);
    }

    /**
     * Logout the given user
     *
     * @param userID the customer to logout
     * @return the login status
     */
    public boolean logout(HttpSession session) {
        session.removeAttribute("ibs");
        return true;
    }

    /**
     * Schedule a Work Order
     *
     * @param assemblyId    Assembly Id
     * @param qty           Original Qty
     * @param location      Location at which to manufacture.
     * @param dueDate       Date when order is due
     * @return woID         Work Order ID
     */
    public WorkOrder scheduleWorkOrder(String assemblyId, int qty, int location, Calendar dueDate) {
        WorkOrderService wos = SpecBeanFactory.getService(WorkOrderService.class);
        return wos.scheduleWorkOrder(assemblyId, qty, location, -1, dueDate);
    }

    /**
     * Schedule a Work Order
     *
     * @param salesId       Sales order id
     * @param oLineId       Order Line ID
     * @param assemblyId    Assembly Id
     * @param qty           Original Qty
     * @param location      Location at which to manufacture.
     * @param dueDate       Date when order is due
     * @return woID         Work Order ID
     */
    public WorkOrder scheduleWorkOrder(int salesId, int oLineId,
            String assemblyId, int qty, int location, Calendar dueDate) {
        WorkOrderService wos = SpecBeanFactory.getService(WorkOrderService.class);
        return wos.scheduleWorkOrder(salesId, oLineId, assemblyId, qty, location, dueDate);
    }

    /**
     * Get Work Order Status
     *
     * @param woID the work order to update
     * @param location the location of work order
     * @return status of operation
     */
    public String getWorkOrderStatus(int woID, int location) {
        WorkOrderService wos = SpecBeanFactory.getService(WorkOrderService.class);
        return wos.findWorkOrder(woID, location).getStatus().toString();
    }

    /**
     * Cancel a Work Order
     *
     * @param woID the work order to cancel
     * @param location the location of work order
     * @return status of operation
     */
    public boolean cancelWorkOrder(int woID, int location) {
        WorkOrderService wos = SpecBeanFactory.getService(WorkOrderService.class);
        return wos.cancelWorkOrder(woID, location);
    }

    /**
     * Complete a Work Order
     *
     * @param woID the work order to complete
     * @param location the location of work order
     * @return status of operation
     */
    public boolean completeWorkOrder(int woID, int location) {
        WorkOrderService service = SpecBeanFactory.getService(WorkOrderService.class);
        return service.completeWorkOrder(woID, location);
    }

    /**
     * Update a Work Order
     *
     * @param woID the work order to update
     * @param location the location of work order
     * @return status of operation
     */
    public boolean updateWorkOrder(int woID, int location) {
        WorkOrderService wos = SpecBeanFactory.getService(WorkOrderService.class);
        return wos.updateWorkOrder(woID, location);
    }

    /**
     * Get All Assembly
     *
     * @return vector of Assembly IDs
     */
    public List<String> getAssemblyIds() {
        MfgService mfg = SpecBeanFactory.getService(MfgService.class);
        return mfg.getAllAssemblyIds();
    }

    public void setAllSupplierURLs(String supplierWebServiceURL,
            String supplierReplyURL) {
        SupplierService supplierService =
                SpecBeanFactory.getService(SupplierService.class);
        supplierService.setAllSupplierURLs(supplierWebServiceURL, supplierReplyURL);
    }

    public Collection<Supplier> getAllSuppliers() {
        SupplierService supplierService =
                SpecBeanFactory.getService(SupplierService.class);
        return supplierService.getAllSuppliers();
    }
}
