/*
 *   SPECjEnterprise2010 - a benchmark for enterprise middleware
 *  Copyright 1995-2010 Standard Performance Evaluation Corporation
 *   All Rights Reserved
 */

package org.spec.jent.ejb.orders.session;

import java.math.BigDecimal;
import java.util.List;
import java.util.Random;
import java.util.Set;

import javax.annotation.Resource;
import javax.ejb.SessionContext;
import javax.ejb.Stateless;
import javax.ejb.TransactionAttribute;
import javax.ejb.TransactionAttributeType;
import javax.persistence.EntityManager;
import javax.persistence.LockModeType;
import javax.persistence.PersistenceContext;

import org.spec.jent.ejb.orders.entity.Customer;
import org.spec.jent.ejb.orders.entity.CustomerInventory;
import org.spec.jent.ejb.orders.entity.CustomerInventoryPK;
import org.spec.jent.ejb.orders.entity.Item;
import org.spec.jent.ejb.shoppingcart.entity.ShoppingCart;

/**
 * Bean implementation class for Enterprise Bean: CustomerSes
 */
@Stateless
public class CustomerSession 
    implements CustomerSessionLocal, CustomerSessionRemote {

    @Resource 
    protected SessionContext mySessionCtx;
    
    @PersistenceContext (unitName="Order")
    protected EntityManager em;
    
    private static final Random rng = new Random(System.currentTimeMillis());

    @TransactionAttribute(TransactionAttributeType.SUPPORTS)
    public Customer getCustomer(Integer userID) {
        return em.find(Customer.class, userID);
    }
    
    public Customer getCustomer(Integer userID, boolean mustExist) {
      Customer customer = em.find(Customer.class, userID);
      if (customer == null) {
        throw new RuntimeException("Customer id=" + userID + " does not exist");
      }
      return customer;
    }
    
    @TransactionAttribute(TransactionAttributeType.SUPPORTS)
    public long countCustomer() {
      return (Long)em.createNamedQuery(Customer.QUERY_COUNT).getSingleResult();
    }

    @TransactionAttribute(TransactionAttributeType.SUPPORTS)
    public boolean validateCustomer(Integer userID) {
      return getCustomer(userID) != null;
    }
    
    public boolean checkCustomerCredit(int custID, BigDecimal orderCost) {
        Customer customer = getCustomer(custID);
        if (customer == null) {
            return false;
        }
        em.lock(customer, LockModeType.READ);
        return customer.hasSufficientCredit(orderCost);
    }
  
  @SuppressWarnings("unchecked")
    @TransactionAttribute(TransactionAttributeType.SUPPORTS)
  public Customer selectCustomerWithGoodCredit(BigDecimal limit) {
      List<Integer> ids = (List<Integer>)
          em.createNamedQuery(Customer.QUERY_BY_CREDIT).setMaxResults(10)
            .setParameter("limit", limit)
            .getResultList();
      Integer random = ids.get(rng.nextInt(ids.size()));
      return getCustomer(random);
  }
  
  @TransactionAttribute(TransactionAttributeType.SUPPORTS)
  public List<CustomerInventory> getInventories(Integer userID) {
        Customer customer = getCustomer(userID);
        if (customer == null) {
            throw new IllegalArgumentException("No customer with id:"+userID);
        }
        return customer.getInventories();
    }

    public CustomerInventory getInventoryItem(Long inventoryID, int custId) {
        return em.find(CustomerInventory.class, new CustomerInventoryPK(inventoryID, custId));
    }

    public boolean sellInventory(Integer userID, Long inventoryID, 
            boolean rollback){

        Customer customer = getCustomer(userID,true);
        CustomerInventory inventory = getInventoryItem(inventoryID, customer.getId());
        if (inventory == null) 
            return false;

        customer.changeBalance(inventory.getTotalCost());
        customer.getInventories().remove(inventory);
        em.remove(inventory);
        
        if (rollback) {
        	mySessionCtx.setRollbackOnly();
        }

        return true;
    }
    
    public void addInventory(Integer custID, ShoppingCart cart, 
            BigDecimal orderCost, boolean rollback) {
        Customer customer = getCustomer(custID);
        Set<Item> items = cart.getItems();
        for (Item item : items) {
            customer.addInventory(item, cart.getQuantity(item),
                    cart.getTotalPrice());
        }
        customer.changeBalance(orderCost.negate());
        
        if (rollback) {
          mySessionCtx.setRollbackOnly();
        }
    }

    // adds completed large order to the current dealership inventory
    public void addInventory(Integer custId, Item item, 
            int quantity, BigDecimal totalValue) {
        Customer customer = getCustomer(custId);
        customer.addInventory(item, quantity, totalValue);
    }

    
}
